<?php
session_start();
include "config/db.php";

if(!isset($_GET['id'])){
    die("Invalid User");
}

$user_id = intval($_GET['id']);

// Fetch main user
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$user = $stmt->get_result()->fetch_assoc();
if(!$user) die("User not found");

// Fetch referrer info (fullname + unique_id)
$referrerData = null;
if (!empty($user['referred_by'])) {
    $r = $conn->prepare("SELECT fullname, unique_id FROM users WHERE id = ?");
    $r->bind_param("i", $user['referred_by']);
    $r->execute();
    $referrerData = $r->get_result()->fetch_assoc();
}

/*
 Build a nested tree up to 3 levels:
  - children = level1 (direct referrals)
  - for each child, children2 = level2
  - for each level2, children3 = level3
*/
function fetch_children($conn, $parent_id) {
    $out = [];
    $q = $conn->prepare("SELECT id, unique_id, fullname, phone, wallet_balance FROM users WHERE referred_by = ? ORDER BY id DESC");
    $q->bind_param("i", $parent_id);
    $q->execute();
    $res = $q->get_result();
    while($row = $res->fetch_assoc()){
        $out[] = $row;
    }
    return $out;
}

// level1
$level1 = fetch_children($conn, $user_id);

// for each level1 node fetch level2 and for each level2 fetch level3
foreach($level1 as &$l1){
    $l1['children'] = fetch_children($conn, $l1['id']);
    foreach($l1['children'] as &$l2){
        $l2['children'] = fetch_children($conn, $l2['id']);
    }
}
unset($l1, $l2);

// COUNT REFERRALS
$direct_count = count($level1);
$level2_count = 0;
$level3_count = 0;

foreach($level1 as $l1){
    $level2_count += count($l1['children']);
    foreach($l1['children'] as $l2){
        $level3_count += count($l2['children']);
    }
}

?>
<?php include "sidebar.php"; ?>

<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>User Profile — Referral Tree</title>
<style>
/* layout to avoid sidebar overlap */
.main-content {
  margin-left:260px;
  padding:22px;
  min-height:100vh;
  background:#f8fafc;
  box-sizing:border-box;
}
@media(max-width:900px){
  .main-content { margin-left:0; padding:14px; }
}

/* Profile card */
.profile-card {
  background:#fff;
  border-radius:12px;
  padding:18px;
  box-shadow:0 8px 30px rgba(2,6,23,0.06);
  max-width:1100px;
  margin:0 auto 20px auto;
  display:flex;
  gap:18px;
  align-items:center;
}
.profile-avatar {
  width:86px;height:86px;border-radius:50%;overflow:hidden;flex:0 0 86px;
  border:4px solid #14b8a6;
}
.profile-avatar img{width:100%;height:100%;object-fit:cover}
.profile-meta h1{margin:0;font-size:20px;color:#0f766e}
.profile-meta p{margin:6px 0;color:#475569}

/* tree container */
.tree-wrap {
  max-width:1200px;
  margin:24px auto;
  background:transparent;
  text-align:center;
}

/* top main node */
.main-node {
  display:inline-block;
  margin:0 auto 18px auto;
  text-align:center;
}
.node-card {
  background:#fff;
  padding:10px 14px;
  border-radius:12px;
  display:inline-block;
  min-width:160px;
  box-shadow:0 8px 30px rgba(2,6,23,0.06);
  transition:transform .18s ease, box-shadow .18s ease;
  cursor:default;
}
.node-card:hover{ transform:translateY(-4px); box-shadow:0 18px 40px rgba(2,6,23,0.09); }
.node-avatar { width:64px;height:64px;border-radius:50%;overflow:hidden;margin:0 auto 8px }
.node-avatar img{width:100%;height:100%;object-fit:cover}
.node-name{font-weight:700;color:#0f172a;font-size:13px}
.node-id{color:#64748b;font-size:12px}

/* levels layout */
.level-row {
  display:flex;
  justify-content:center;
  gap:26px;
  flex-wrap:wrap;
  margin-bottom:28px;
  align-items:flex-start;
}

/* each tree node (clickable) */
.tree-node {
  text-align:center;
  width:160px;
  position:relative;
}
.tree-node .node-card.clickable{ cursor:pointer; }
.node-children {
  margin-top:12px;
  display:none; /* collapsed by default */
  transition:all .25s ease;
}

/* connector lines */
.connector-vertical {
  width:2px;height:18px;background:linear-gradient(#14b8a6,#0f766e);
  margin:0 auto;
}
.connector-horizontal {
  height:2px;background:linear-gradient(#14b8a6,#0f766e);
  position:absolute; left:50%;
  top:-10px; transform-origin:left center;
}

/* small label under node */
.node-sub { font-size:12px;color:#64748b;margin-top:6px }

/* show children when .open applied */
.tree-node.open > .node-children { display:block; }

/* animations */
.fade-in { animation: fadeIn .35s ease both; }
@keyframes fadeIn { from{opacity:0; transform:translateY(8px)} to{opacity:1; transform:none} }

/* responsive adjustments */
@media(max-width:640px){
  .node-card{min-width:140px}
  .tree-node{width:140px}
  .profile-card{flex-direction:column;align-items:flex-start}
  .profile-avatar{width:72px;height:72px}
}

/* small actions */
.action-btn {
  display:inline-block;padding:6px 10px;border-radius:8px;background:#0f766e;color:#fff;text-decoration:none;font-size:13px;margin-top:8px;
}
</style>
</head>
<body>

<div class="main-content">

  <div class="profile-card">
    <div class="profile-avatar">
      <img src="assets/user.png" alt="avatar">
    </div>
    <div class="profile-meta">
      <h1><?= htmlspecialchars($user['fullname']) ?> <small style="font-weight:600;color:#0f766e"> (<?= htmlspecialchars($user['unique_id']) ?>)</small></h1>
      <p><strong>Phone:</strong> <?= htmlspecialchars($user['phone']) ?> &nbsp;•&nbsp; <strong>State:</strong> <?= htmlspecialchars($user['state']) ?></p>
      <p><strong>Wallet:</strong> ₦<?= number_format($user['wallet_balance'],2) ?></p>
      <p><strong>Referrer:</strong>
        <?php if ($referrerData): ?>
          <?= htmlspecialchars($referrerData['fullname']) ?> (<?= htmlspecialchars($referrerData['unique_id']) ?>)
        <?php else: ?>
          None
        <?php endif; ?>
      </p>
      <a class="action-btn" href="admin_register.php">Register Member</a>
      <a class="action-btn" href="admin_users.php" style="background:#334155;margin-left:8px">All Members</a>
      <div style="margin-top:15px; display:flex; gap:20px; flex-wrap:wrap;">
    <div style="background:#0f766e; color:white; padding:8px 14px; border-radius:8px;">
        <strong>Direct:</strong> <?= $direct_count ?>
    </div>

    <div style="background:#0369a1; color:white; padding:8px 14px; border-radius:8px;">
        <strong>Level 2:</strong> <?= $level2_count ?>
    </div>

    <div style="background:#7c3aed; color:white; padding:8px 14px; border-radius:8px;">
        <strong>Level 3:</strong> <?= $level3_count ?>
    </div>
</div>

    </div>
  </div>

  <div class="tree-wrap">

    <!-- main node -->
    <div class="main-node fade-in">
      <div class="node-card">
        <div class="node-avatar"><img src="assets/user.png" alt=""></div>
        <div class="node-name"><?= htmlspecialchars($user['fullname']) ?></div>
        <div class="node-id"><?= htmlspecialchars($user['unique_id']) ?></div>
      </div>
    </div>

    <!-- connector to level1 -->
    <div class="connector-vertical" style="margin-bottom:18px"></div>

    <!-- LEVEL 1 ROW -->
    <div class="level-row" id="level1-row">
      <?php if(count($level1) == 0): ?>
        <p style="color:#64748b">This user has no direct referrals yet.</p>
      <?php endif; ?>

      <?php foreach($level1 as $l1): ?>
        <div class="tree-node fade-in" data-id="<?= $l1['id'] ?>">

          <div class="node-card clickable" onclick="toggleNode(this)">
            <div class="node-avatar"><img src="assets/user.png" alt=""></div>
            <div class="node-name"><?= htmlspecialchars($l1['fullname']) ?></div>
            <div class="node-id"><?= htmlspecialchars($l1['unique_id']) ?></div>
            <div class="node-sub">Direct</div>
          </div>

          <!-- small vertical connector -->
          <div class="connector-vertical" style="height:14px;margin-top:8px"></div>

          <!-- children container (level2 for this node) -->
          <div class="node-children">
            <!-- populate level2 nodes for this L1 -->
            <?php if(!empty($l1['children'])): ?>
              <div style="display:flex;justify-content:center;gap:14px;flex-wrap:wrap;margin-top:8px">
                <?php foreach($l1['children'] as $l2): ?>
                  <div class="tree-node" data-id="<?= $l2['id'] ?>">
                    <div class="node-card clickable" onclick="toggleNode(this)">
                      <div class="node-avatar"><img src="assets/user.png" alt=""></div>
                      <div class="node-name"><?= htmlspecialchars($l2['fullname']) ?></div>
                      <div class="node-id"><?= htmlspecialchars($l2['unique_id']) ?></div>
                      <div class="node-sub">Level 2</div>
                    </div>

                    <div class="connector-vertical" style="height:12px;margin-top:8px"></div>

                    <div class="node-children">
                      <?php if(!empty($l2['children'])): ?>
                        <div style="display:flex;justify-content:center;gap:12px;flex-wrap:wrap;margin-top:8px">
                          <?php foreach($l2['children'] as $l3): ?>
                            <div class="tree-node">
                              <div class="node-card">
                                <div class="node-avatar"><img src="assets/user.png" alt=""></div>
                                <div class="node-name"><?= htmlspecialchars($l3['fullname']) ?></div>
                                <div class="node-id"><?= htmlspecialchars($l3['unique_id']) ?></div>
                                <div class="node-sub">Level 3</div>
                              </div>
                            </div>
                          <?php endforeach; ?>
                        </div>
                      <?php else: ?>
                        <!-- no level3 children -->
                        <div style="margin-top:8px;color:#94a3b8">No deeper referrals</div>
                      <?php endif; ?>
                    </div>
                  </div>
                <?php endforeach; ?>
              </div>
            <?php else: ?>
              <div style="margin-top:10px;color:#94a3b8">No level 2 referrals under this member</div>
            <?php endif; ?>
          </div>
        </div>
      <?php endforeach; ?>
    </div>

  </div>
</div>

<script>
// Toggle open/close on clickable node cards
function toggleNode(el){
  // el is .node-card; parent .tree-node should toggle class 'open'
  const node = el.closest('.tree-node');
  if(!node) return;

  // collapse other open children at same level for clarity (optional)
  const siblings = node.parentElement.querySelectorAll(':scope > .tree-node.open');
  siblings.forEach(s => { if(s !== node) s.classList.remove('open'); });

  node.classList.toggle('open');

  // smooth scroll into view when expanded on mobile
  if(node.classList.contains('open')){
    setTimeout(()=> node.scrollIntoView({behavior:'smooth', block:'center'}), 120);
  }
}
</script>

</body>
</html>
